--[[
    UnloadPotatoSeeder

	Enables Vehicles capable of loading silage additive to unload it.

	@author: 	BayernGamers
	@date: 		20.05.2025
	@version:	1.0

	History:	v1.0 @20.05.2025 - initial implementation in FS25
				-------------------------------------------------------------------------------------------

	License:    Terms:
                    Usage:
                        Feel free to use this work as-is as long as you adhere to the following terms:
                    Attribution:
                        You must give appropriate credit to the original author when using this work.
                    No Derivatives:
                        You may not alter, transform, or build upon this work in any way.
                    Usage:
                        The work may be used for personal and commercial purposes, provided it is not modified or adapted.
                    Additional Clause:
                        This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/events/UnloadTypeChangeEvent.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "UnloadPotatoSeeder.lua")

UnloadPotatoSeeder = {}
UnloadPotatoSeeder.MOD_DIR = g_currentModDirectory
UnloadPotatoSeeder.MOD_NAME = g_currentModName

function UnloadPotatoSeeder.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(FillUnit, specializations) and SpecializationUtil.hasSpecialization(SowingMachine, specializations)
end

function UnloadPotatoSeeder.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onReadStream", UnloadPotatoSeeder)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", UnloadPotatoSeeder)
    SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", UnloadPotatoSeeder)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", UnloadPotatoSeeder)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", UnloadPotatoSeeder)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", UnloadPotatoSeeder)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", UnloadPotatoSeeder)
end

function UnloadPotatoSeeder.registerFunction(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "toggleUnloadFillType", UnloadPotatoSeeder.toggleUnloadFillType)
end

function UnloadPotatoSeeder.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "unloadFillUnits", UnloadPotatoSeeder.unloadFillUnits)
end

function UnloadPotatoSeeder:onReadStream(streamId, connection)
    if connection:getIsServer() then
        local spec = self.spec_unloadPotatoSeeder

        spec.unloadFillTypeIndex = streamReadUIntN(streamId, FillTypeManager.SEND_NUM_BITS)
    end
end

function UnloadPotatoSeeder:onWriteStream(streamId, connection)
    if not connection:getIsServer() then
        local spec = self.spec_unloadPotatoSeeder

        streamWriteUIntN(streamId, spec.unloadFillTypeIndex, FillTypeManager.SEND_NUM_BITS)
    end
end

function UnloadPotatoSeeder:onPreLoad()
    self.spec_unloadPotatoSeeder = {}
end

function UnloadPotatoSeeder:onLoad(savegame)
    local spec = self.spec_unloadPotatoSeeder
    spec.actionEvents = {}

    if self:getFillUnitSupportsFillType(index, FillType.POTATO) then
        spec.unloadFillTypeIndex = FillType.POTATO
    else
        spec.unloadFillTypeIndex = FillType.SEEDS
    end
end

function UnloadPotatoSeeder:onPostLoad(savegame)
	local spec_fillUnit = self.spec_fillUnit

	if spec_fillUnit ~= nil then
		if spec_fillUnit.unloading == nil then
			spec_fillUnit.unloading = {}
		end

		if #spec_fillUnit.unloading == 0 then
			local unloadNodeBack = createTransformGroup("unloadNodeBack")
            local unloadNodeLeft = createTransformGroup("unloadNodeLeft")
            local unloadNodeRight = createTransformGroup("unloadNodeRight")
			
			for index, component in ipairs(self.components) do
				log:printDevInfo("Found Component: " .. tostring(index) .. " with node: " .. tostring(component.node), LoggingUtil.DEBUG_LEVELS.HIGH)
				if component.node ~= nil then
					link(component.node, unloadNodeBack)
                    link(component.node, unloadNodeLeft)
                    link(component.node, unloadNodeRight)
					break
				end
			end

			local lengthOffset = self.xmlFile:getValue("vehicle.base.size#length")
            local widthOffset = self.xmlFile:getValue("vehicle.base.size#width")
			if lengthOffset == nil then
				lengthOffset = 5
			else
				lengthOffset = lengthOffset * 0.5 + 0.75
			end

            if widthOffset == nil then
                widthOffset = 5
            else
                widthOffset = widthOffset * 0.5 + 1.5
            end
			
			log:printDevInfo("LengthOffset: -" .. tostring(lengthOffset), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("WidthOffset: " .. tostring(widthOffset), LoggingUtil.DEBUG_LEVELS.HIGH)

			setTranslation(unloadNodeBack, 0, 0.2, -lengthOffset)
            setTranslation(unloadNodeLeft, widthOffset, 0.2, 0)
            setTranslation(unloadNodeRight, -widthOffset, 0.2, 0)

            setRotation(unloadNodeLeft, 0, math.rad(90), 0)
            setRotation(unloadNodeRight, 0, math.rad(90), 0)

			local unloadingBack =  {
				node = unloadNodeBack,
				offset = {0, 0, 0},
				width = 6
			}

            local unloadingLeft =  {
                node = unloadNodeLeft,
                offset = {0, 0, 0},
                width = 10
            }

            local unloadingRight =  {
                node = unloadNodeRight,
                offset = {0, 0, 0},
                width = 10
            }

            table.insert(spec_fillUnit.unloading, unloadingRight)
            table.insert(spec_fillUnit.unloading, unloadingBack)
            table.insert(spec_fillUnit.unloading, unloadingLeft)

			log:printDevInfo("Created Unloading Node Back " .. tostring(unloading), LoggingUtil.DEBUG_LEVELS.HIGH)
			log:printDevInfo("Values: " .. tostring(unloadingBack.node) .. " " .. tostring(unloadingBack.offset) .. " " .. tostring(unloadingBack.width), LoggingUtil.DEBUG_LEVELS.HIGH)

            log:printDevInfo("Created Unloading Node Left " .. tostring(unloading), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Values: " .. tostring(unloadingLeft.node) .. " " .. tostring(unloadingLeft.offset) .. " " .. tostring(unloadingLeft.width), LoggingUtil.DEBUG_LEVELS.HIGH)

            log:printDevInfo("Created Unloading Node Right " .. tostring(unloading), LoggingUtil.DEBUG_LEVELS.HIGH)
            log:printDevInfo("Values: " .. tostring(unloadingRight.node) .. " " .. tostring(unloadingRight.offset) .. " " .. tostring(unloadingRight.width), LoggingUtil.DEBUG_LEVELS.HIGH)
		end
	end
end

function UnloadPotatoSeeder:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
     local spec = self.spec_unloadPotatoSeeder
    if spec.actionEvents == nil then
        spec.actionEvents = {}
    else
        self:clearActionEventsTable(spec.actionEvents)
    end

    if isActiveForInputIgnoreSelection then
        local _, toggleUnloadFillTypeActionEventId = self:addActionEvent(spec.actionEvents, InputAction.UP_toggleUnloadFillType, self, UnloadPotatoSeeder.toggleUnloadFillType, false, true, false, true)
        g_inputBinding:setActionEventTextPriority(toggleUnloadFillTypeActionEventId, GS_PRIO_NORMAL)
        log:printDevInfo("Added Toggle unload fillType ActionEvent", LoggingUtil.DEBUG_LEVELS.HIGH)
    end
end

function UnloadPotatoSeeder:onUpdate(dt)
    local spec = self.spec_unloadPotatoSeeder
    local spec_fillUnit = self.spec_fillUnit

    if spec_fillUnit ~= nil then
        local supportsFillType = false

        for index, fillUnit in ipairs(spec_fillUnit.fillUnits) do
            if self:getFillUnitSupportsFillType(index, FillType.POTATO) then
                if self:getFillUnitFillLevel(index) > 0 then
                    fillUnit.canBeUnloaded = true
                    supportsFillType = true
                else
                    fillUnit.canBeUnloaded = false
                end

                local fillType = g_fillTypeManager:getFillTypeByIndex(spec.unloadFillTypeIndex)
                g_currentMission:addExtraPrintText(g_i18n:getText("info_unloadingFillType"):format(fillType.title))
            end
        end

        g_inputBinding:setActionEventActive(spec.actionEvents[InputAction.UP_toggleUnloadFillType], supportsFillType)
    end
end

function UnloadPotatoSeeder:unloadFillUnits(superFunc, silentMode)
    if self.isServer then
        local spec = self.spec_unloadPotatoSeeder
        local spec_fillUnit = self.spec_fillUnit

        if not spec_fillUnit.unloadingFillUnitsRunning then
            spec_fillUnit.unloadingFillUnitsRunning = true
            
            local unloadingNodes = spec_fillUnit.unloading
            local placableAreas = {}

            for _, unloadNode in ipairs(unloadingNodes) do
                local node = unloadNode.node
                local offset = unloadNode.offset
                local offsetX, offsetY, offsetZ = unpack(offset)
                local posX, posY, posZ = localToWorld(node, offsetX - unloadNode.width * 0.5, offsetY, offsetZ)
                local placableArea = {
                    ["startX"] = posX,
                    ["startY"] = posY,
                    ["startZ"] = posZ
                }
                local rotX, rotY, rotZ = getWorldRotation(node)

                placableArea.rotX = rotX
                placableArea.rotY = rotY
                placableArea.rotZ = rotZ

                local dirX, dirY, dirZ = localDirectionToWorld(node, 1, 0, 0)

                placableArea.dirX = dirX
                placableArea.dirY = dirY
                placableArea.dirZ = dirZ

                local perpDirX, perpDirY, perpDirZ = localDirectionToWorld(node, 0, 0, 1)

                placableArea.dirPerpX = perpDirX
                placableArea.dirPerpY = perpDirY
                placableArea.dirPerpZ = perpDirZ
                placableArea.yOffset = 1
                placableArea.maxWidth = math.huge
                placableArea.maxLength = math.huge
                placableArea.maxHeight = math.huge
                placableArea.width = unloadNode.width

                table.insert(placableAreas, placableArea)
            end

            local palletsToUnload = {}
            local occupiedAreas = {}
            local receiverVehicles = {}

            for fillUnitIndex, fillUnit in ipairs(self:getFillUnits()) do
                local fillLevel = self:getFillUnitFillLevel(fillUnitIndex)

                if fillUnit.canBeUnloaded and self:getFillUnitFillLevel(fillUnitIndex) > 0 then
                    local unloadFillType = g_fillTypeManager:getFillTypeByIndex(spec.unloadFillTypeIndex)
                    local palletFilename = unloadFillType.palletFilename

                    if palletFilename ~= nil then
                        local palletData = {
                            ["fillUnitIndex"] = fillUnitIndex,
                            ["fillTypeIndex"] = spec.unloadFillTypeIndex,
                            ["fillLevel"] = fillLevel,
                            ["filename"] = palletFilename
                        }

                        table.insert(palletsToUnload, palletData)
                    end
                end
            end
            if self:getFillUnitHasMountedPalletsToUnload() then
                local mountedPallets = self:getFillUnitMountedPalletsToUnload()

                if mountedPallets ~= nil and #mountedPallets > 0 then
                    for _, pallet in ipairs(mountedPallets) do
                        local posX, posY, posZ, area, areaSize, _ = PlacementUtil.getPlace(placableAreas, pallet.size, occupiedAreas, true, true, false, true)

                        if posX == nil then
                            if silentMode == nil or not silentMode then
                                g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO, spec_fillUnit.texts.unloadNoSpace)
                            end
                            g_server:broadcastEvent(FillUnitUnloadedEvent.new(self, nil, true), false, nil, self)
                        else
                            if pallet.unmount ~= nil then
                                pallet:unmount()
                            end
                            PlacementUtil.markPlaceUsed(occupiedAreas, area, areaSize)
                            pallet:setAbsolutePosition(posX, posY, posZ, 0, MathUtil.getYRotationFromDirection(area.dirPerpX, area.dirPerpZ), 0)
                            SpecializationUtil.raiseEvent(self, "onFillUnitUnloadPallet", pallet)
                            g_server:broadcastEvent(FillUnitUnloadedEvent.new(self, pallet), false, nil, self)
                        end
                    end
                end
            end
            local function processNextPallet()
                -- upvalues: palletsToUnload, receiverVehicles, self, processNextPallet, silentMode, spec_fillUnit, placableAreas, occupiedAreas
                local currentPallet = palletsToUnload[1]

                if currentPallet == nil then
                    SpecializationUtil.raiseEvent(self, "onFillUnitUnloaded", true)
                    g_server:broadcastEvent(FillUnitUnloadedEvent.new(self, nil, false, true), false, nil, self)
                    spec_fillUnit.unloadingFillUnitsRunning = false
                    return
                else
                    for receiver, _ in pairs(receiverVehicles) do
                        local targetFillUnitIndex = receiver:getFirstValidFillUnitToFill(currentPallet.fillTypeIndex)
                        if targetFillUnitIndex ~= nil then
                            local transferredAmount = receiver:addFillUnitFillLevel(self:getOwnerFarmId(), targetFillUnitIndex, currentPallet.fillLevel, currentPallet.fillTypeIndex, ToolType.UNDEFINED, nil)

                            local fillTypeToRemove = self:getFillUnitFillType(currentPallet.fillUnitIndex)
                            self:addFillUnitFillLevel(self:getOwnerFarmId(), currentPallet.fillUnitIndex, -transferredAmount, fillTypeToRemove, ToolType.UNDEFINED, nil)
                            currentPallet.fillLevel = currentPallet.fillLevel - transferredAmount
                            if receiver:getFillUnitFreeCapacity(targetFillUnitIndex) <= 0 then
                                receiverVehicles[receiver] = nil
                            end
                        end
                    end
                    if currentPallet.fillLevel > 0 then
                        local function onVehicleLoaded(_, loadedVehicles, loadingState, _)
                            -- upvalues: receiverVehicles, self, processNextPallet, silentMode, spec_fillUnit
                            if loadingState == VehicleLoadingState.OK then
                                for _, loadedVehicle in ipairs(loadedVehicles) do
                                    loadedVehicle:emptyAllFillUnits(true)
                                    receiverVehicles[loadedVehicle] = true
                                    SpecializationUtil.raiseEvent(self, "onFillUnitUnloadPallet", loadedVehicle)
                                end
                                processNextPallet()
                            elseif loadingState == VehicleLoadingState.NO_SPACE then
                                if silentMode == nil or not silentMode then
                                    g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO, spec_fillUnit.texts.unloadNoSpace)
                                end
                                SpecializationUtil.raiseEvent(self, "onFillUnitUnloaded", false)
                                g_server:broadcastEvent(FillUnitUnloadedEvent.new(self, nil, true, false), false, nil, self)
                                spec_fillUnit.unloadingFillUnitsRunning = false
                            end
                        end
                        local loadingData = VehicleLoadingData.new()
                        loadingData:setFilename(currentPallet.filename)
                        loadingData:setLoadingPlace(placableAreas, occupiedAreas, 0.5, true)
                        loadingData:setPropertyState(VehiclePropertyState.OWNED)
                        loadingData:setOwnerFarmId(self:getOwnerFarmId())
                        loadingData:load(onVehicleLoaded)
                    else
                        table.remove(palletsToUnload, 1)
                        processNextPallet()
                    end
                end
            end
            processNextPallet()
        end
    else
        g_client:getServerConnection():sendEvent(FillUnitUnloadEvent.new(self))
        return
    end
end

function UnloadPotatoSeeder:toggleUnloadFillType()
    local spec = self.spec_unloadPotatoSeeder

    if spec.unloadFillTypeIndex == FillType.POTATO then
        spec.unloadFillTypeIndex = FillType.SEEDS
    else
        spec.unloadFillTypeIndex = FillType.POTATO
    end

    if spec.unloadFillTypeIndex ~= FillType.POTATO and spec.unloadFillTypeIndex ~= FillType.SEEDS then
        spec.unloadFillTypeIndex = FillType.POTATO
    end

    UnloadTypeChangeEvent.sendEvent(self, spec.unloadFillTypeIndex, false)
end