-- FillableBucketFix.lua
--
-- Fix für Eimer / Hand-Objekte mit FillUnit + Dashboard-Animation:
-- Stellt nach dem Laden eines Spielstands sicher, dass die Füllanimation
-- (fillLevelAnimation) zur tatsächlichen Füllmenge passt.

FillableBucketFix = {}
FillableBucketFix.MOD_NAME = g_currentModName or "unknownMod"

local FillableBucketFix_mt = Class(FillableBucketFix)

--- Registrierung als Spezialisierung (wird von modDesc.xml aufgerufen)
function FillableBucketFix.prerequisitesPresent(specializations)
    -- Wir brauchen die FillUnit-Spezialisierung
    return SpecializationUtil.hasSpecialization(FillUnit, specializations)
end

function FillableBucketFix.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", FillableBucketFix)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", FillableBucketFix)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", FillableBucketFix)
end

--- onLoad: Flag setzen, dass wir nach dem Laden einmalig initialisieren wollen
function FillableBucketFix:onLoad(savegameId)
    local spec = self.spec_fillableBucketFix
    if spec == nil then
        spec = {}
        self.spec_fillableBucketFix = spec
    end

    -- Wir initialisieren auf dem Client, nachdem der Netzwerk-Status da ist
    spec.needsInitAfterLoad = true
end

--- onPostLoad: hier könnte man auch initialisieren, aber um sicherzugehen,
--- dass FillLevel & Co. wirklich synchron sind, warten wir auf onUpdate.
function FillableBucketFix:onPostLoad(savegameId)
    -- Hier aktuell nichts, aber Platz für spätere Erweiterungen
end

--- onUpdate: einmalig nach dem Laden die Animation auf den FillLevel setzen
function FillableBucketFix:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    if not self.isClient then
        return
    end

    local spec = self.spec_fillableBucketFix
    if spec == nil or not spec.needsInitAfterLoad then
        return
    end

    -- Wir versuchen genau einmal zu initialisieren
    spec.needsInitAfterLoad = false

    local specFillUnit = self.spec_fillUnit
    if specFillUnit == nil or specFillUnit.fillUnits == nil then
        return
    end

    -- passenden FillUnit suchen:
    -- - bevorzugt den, der tatsächlich Füllstand > 0 hat
    -- - ansonsten einfach den ersten gültigen
    local chosenFillUnit = nil
    local chosenIndex = nil

    for index, fillUnit in ipairs(specFillUnit.fillUnits) do
        if fillUnit.capacity ~= nil and fillUnit.capacity > 0 then
            local level = fillUnit.fillLevel or 0

            if chosenFillUnit == nil then
                chosenFillUnit = fillUnit
                chosenIndex = index
            end

            if level > 0.001 then
                chosenFillUnit = fillUnit
                chosenIndex = index
                break
            end
        end
    end

    if chosenFillUnit == nil then
        return
    end

    local level = chosenFillUnit.fillLevel or 0
    local capacity = chosenFillUnit.capacity or 0
    local norm = 0

    if capacity > 0 then
        norm = level / capacity
    end

    -- Clamp auf [0..1], damit nix aus dem Ruder läuft
    if norm < 0 then
        norm = 0
    elseif norm > 1 then
        norm = 1
    end

    -- Animationsname muss zu deinem XML passen
    local animationName = "fillLevelAnimation"

    if self.setAnimationTime ~= nil then
        -- Debug-Log (bei Bedarf auskommentieren)
        -- Logging.info("[%s:FillableBucketFix] Init animation '%s' for fillUnit index %d: level=%.3f / %.3f (norm=%.3f)",
        --     FillableBucketFix.MOD_NAME, animationName, chosenIndex, level, capacity, norm)

        self:setAnimationTime(animationName, norm, true, true)
    else
        print(string.format("[%s:FillableBucketFix] WARNING: setAnimationTime not available on this vehicle type!", FillableBucketFix.MOD_NAME))
    end
end
