-- ============================================================================
-- === Darkernights.Lua
-- === Mod by [LSMT] Modding Team 
-- === LS25 / FS25
-- === Script by [LSMT] BaTt3RiE @ 2025
-- === Ver 1.1.0.0 (Optimiert, MP-Support)
-- ============================================================================

Darkernights = {}

local ModDirectory = g_currentModDirectory

local DEBUG = false -- Setze auf true, um Debug-Logs zu aktivieren

local function debugPrint(...)
    if DEBUG then
        print(...)
    end
end

DarkernightsEvent = {}
DarkernightsEvent_mt = Class(DarkernightsEvent, Event)
InitEventClass(DarkernightsEvent, "DarkernightsEvent")

function DarkernightsEvent:emptyNew()
    return Event.new(DarkernightsEvent_mt)
end

function DarkernightsEvent:new(isNightMode)
    local event = DarkernightsEvent:emptyNew()
    event.isNightMode = isNightMode
    return event
end

function DarkernightsEvent:writeStream(streamId, connection)
    streamWriteBool(streamId, self.isNightMode)
end

function DarkernightsEvent:readStream(streamId, connection)
    self.isNightMode = streamReadBool(streamId)
    self:run(connection)
end

function DarkernightsEvent:run(connection)
    -- Anwenden der Lighting-Curves auch bei nachträglichem Joinen
    Darkernights:applyCustomLightingCurves()
end

function DarkernightsEvent.sendEvent(isNightMode, noEventSend)
    if noEventSend then return end
    
    local event = DarkernightsEvent:new(isNightMode)
    if g_server ~= nil then
        g_server:broadcastEvent(event)
    else
        g_client:getServerConnection():sendEvent(event)
    end
end

-- Default Settings (Fallback)
Darkernights.settings = {
    slope = 1.2,
    toe = 0.4,
    shoulder = 0.8,
    blackClip = 0.00,
    whiteClip = 0.04,
    bloomThreshold = 2.0,
    bloomMagnitude = 0.0
}

-- XML laden
function Darkernights:loadSettingsFromXML()
    local configFilePath = ModDirectory .. "darkernightsSettings.xml"
    local xmlFile = loadXMLFile("DarkernightsConfig", configFilePath)

    if xmlFile ~= 0 then
        self.settings.slope       = getXMLFloat(xmlFile, "darkernightsSettings.toneMapping#slope") or self.settings.slope
        self.settings.toe         = getXMLFloat(xmlFile, "darkernightsSettings.toneMapping#toe") or self.settings.toe
        self.settings.shoulder    = getXMLFloat(xmlFile, "darkernightsSettings.toneMapping#shoulder") or self.settings.shoulder
        self.settings.blackClip   = getXMLFloat(xmlFile, "darkernightsSettings.toneMapping#blackClip") or self.settings.blackClip
        self.settings.whiteClip   = getXMLFloat(xmlFile, "darkernightsSettings.toneMapping#whiteClip") or self.settings.whiteClip

        self.settings.bloomThreshold = getXMLFloat(xmlFile, "darkernightsSettings.bloom#threshold") or self.settings.bloomThreshold
        self.settings.bloomMagnitude = getXMLFloat(xmlFile, "darkernightsSettings.bloom#magnitude") or self.settings.bloomMagnitude

        delete(xmlFile)
        debugPrint("[Darkernights] XML Settings erfolgreich geladen:", configFilePath)
    else
        debugPrint("[Darkernights] XML nicht gefunden, nutze Standardwerte.")
    end
end

-- Custom Lighting Curves
Darkernights.customCurves = {
    primaryExtraterrestrialColor = {
        {time = 0.0,  values = {0, 0, 0}},
        {time = 5.0,  values = {0, 0, 0}},
        {time = 6.5,  values = {1, 1, 1}},
        {time = 18.0, values = {1, 1, 1}},
        {time = 19.5, values = {0, 0, 0}},
        {time = 24.0, values = {0, 0, 0}}
    },
    moonBrightnessScale = {
        {time = 0.0,  values = {0}},
        {time = 7.0,  values = {0}},
        {time = 19.0, values = {0}},
        {time = 24.0, values = {0}}
    },
    primaryDynamicLightingScale = {
        {time = 0.0,  values = {0}},
        {time = 5.0,  values = {0}},
        {time = 6.5,  values = {1}},
        {time = 18.0, values = {1}},
        {time = 19.5, values = {0}},
        {time = 20.0, values = {0}}
    },
    secondaryExtraterrestrialColor = {
        {time = 0.0,  values = {0, 0, 0}},
        {time = 5.0,  values = {0, 0, 0}},
        {time = 6.5,  values = {1, 1, 1}},
        {time = 18.0, values = {1, 1, 1}},
        {time = 19.5, values = {0, 0, 0}},
        {time = 24.0, values = {0, 0, 0}}
    },
    groundAlbedo = {
        {time = 0.0,  values = {0.0, 0.0, 0.0}},
        {time = 6.0,  values = {0.0, 0.0, 0.0}},
        {time = 7.0,  values = {0.2, 0.2, 0.2}},
        {time = 18.0, values = {0.2, 0.2, 0.2}},
        {time = 19.5, values = {0.0, 0.0, 0.0}},
        {time = 24.0, values = {0.0, 0.0, 0.0}}
    }
}

Darkernights.originalCurves = {
    primaryExtraterrestrialColorCurveData = nil,
    moonBrightnessScaleCurveData = nil,
    primaryDynamicLightingScaleCurveData = nil,
    secondaryExtraterrestrialColorCurveData = nil,
    groundAlbedoCurveData = nil
}

function Darkernights:convertCurveFormat(curve)
    local result = {}
    for _, point in ipairs(curve) do
        table.insert(result, {point.time, point.values})
    end
    return result
end

function Darkernights:applyCustomLightingCurves()
    if not g_currentMission or not g_currentMission.environment or not g_currentMission.environment.lighting then
        return false
    end
    
    local lighting = g_currentMission.environment.lighting
    if not self.originalCurves.primaryExtraterrestrialColorCurveData and lighting.primaryExtraterrestrialColorCurveData then
        self.originalCurves.primaryExtraterrestrialColorCurveData = lighting.primaryExtraterrestrialColorCurveData
        self.originalCurves.moonBrightnessScaleCurveData = lighting.moonBrightnessScaleCurveData
        self.originalCurves.primaryDynamicLightingScaleCurveData = lighting.primaryDynamicLightingScaleCurveData
        self.originalCurves.secondaryExtraterrestrialColorCurveData = lighting.secondaryExtraterrestrialColorCurveData
        self.originalCurves.groundAlbedoColorCurveData = lighting.groundAlbedoColorCurveData
    end
    
    lighting.primaryExtraterrestrialColorCurveData = self:convertCurveFormat(self.customCurves.primaryExtraterrestrialColor)
    lighting.moonBrightnessScaleCurveData = self:convertCurveFormat(self.customCurves.moonBrightnessScale)
    lighting.primaryDynamicLightingScaleCurveData = self:convertCurveFormat(self.customCurves.primaryDynamicLightingScale)
    lighting.secondaryExtraterrestrialColorCurveData = self:convertCurveFormat(self.customCurves.secondaryExtraterrestrialColor)
    lighting.groundAlbedoColorCurveData = self:convertCurveFormat(self.customCurves.groundAlbedo)
    
    lighting:updateCurves()
    return true
end

function Darkernights:restoreOriginalLightingCurves()
    if not g_currentMission or not g_currentMission.environment or not g_currentMission.environment.lighting then
        return
    end

    local lighting = g_currentMission.environment.lighting
    if self.originalCurves.primaryExtraterrestrialColorCurveData then
        lighting.primaryExtraterrestrialColorCurveData = self.originalCurves.primaryExtraterrestrialColorCurveData
        lighting.moonBrightnessScaleCurveData = self.originalCurves.moonBrightnessScaleCurveData
        lighting.primaryDynamicLightingScaleCurveData = self.originalCurves.primaryDynamicLightingScaleCurveData
        lighting.secondaryExtraterrestrialColorCurveData = self.originalCurves.secondaryExtraterrestrialColorCurveData
        lighting.groundAlbedoColorCurveData = self.originalCurves.groundAlbedoColorCurveData
        lighting:updateCurves()
    end
end

function Darkernights:loadMap(name)
    print("Darkernights Mod wird geladen")

    -- Settings aus XML laden
    self:loadSettingsFromXML()

    -- Tone-Mapping Werte anwenden
    if g_currentMission and g_currentMission.environment then
        setToneMappingCurveSlope(self.settings.slope)
        setToneMappingCurveToe(self.settings.toe)
        setToneMappingCurveShoulder(self.settings.shoulder)
        setToneMappingCurveBlackClip(self.settings.blackClip)
        setToneMappingCurveWhiteClip(self.settings.whiteClip)
        setBloomMaskThreshold(self.settings.bloomThreshold)
        setBloomMagnitude(self.settings.bloomMagnitude)
    end

    -- Lighting-Kurven anwenden
    self:applyCustomLightingCurves()

    -- Server synchronisiert Nachtstatus für Clients
    if g_currentMission:getIsServer() then
        local isNight = self:isNightTime()
        DarkernightsEvent.sendEvent(isNight)
    end
end

function Darkernights:deleteMap()
    if g_currentMission then
        g_currentMission:removeUpdateable(self)
    end
    self:restoreOriginalLightingCurves()
end

-- Nachtzeit-Prüfung für Initial-Sync
function Darkernights:isNightTime()
    local env = g_currentMission.environment
    if not env or not env.daylight then return false end

    local dayTimeMinutes = env.dayTime / 60000
    return (env.daylight.logicalNightStartMinutes <= dayTimeMinutes or 
            dayTimeMinutes < env.daylight.logicalNightEndMinutes)
end

addModEventListener(Darkernights)
